#ifndef __IAudioFileFormat__
#define __IAudioFileFormat__

#include <Collections/CollectionTypedef.h>
#include <Basics/CSystemString.hpp>
#include <IO/CFileStream.hpp>
#include "TAudioBuffer.hpp"
using Exponent::Audio::TAudioBuffer;
using Exponent::IO::CFileStream;
using Exponent::Basics::CSystemString;
using Exponent::Collections::TStringCountedPointerArray;

//	===========================================================================

namespace Exponent
{
	namespace Audio
	{

//	===========================================================================

		/**
		 * @interface IAudioFileFormat IAudioFileFormat.hpp
		 * @brief Interface to an audio file format
		 *
		 * The IAudioFileFormat defines a template interface that should inherited from\n
		 * during the creation of actual file formats, such as WAVE, AIFF etc.\n
		 * In use the file format is somewhat complex, but made clearer by looking at code\n
		 * Below is an example of usage:\n
		 * \n
		 * READING\n
		 * \n
		 * @code
		 * // First we open the file
		 * TWaveFileFormat<double> waveFileFormat;
		 * if (!waveFileFormat.openFile(CFileStream::e_input, myFileName))
		 * {
		 *    printf("Failed to open file");
		 *    return;
		 * }
		 *
		 * // Now we read in the format header
		 * TWaveFileFormat<double>::SAudioFileFormat format;
		 * if (!waveFileFormat.readHeader(format))
		 * {
		 *     printf("Failed to read header");
		 *     return;
		 * }
		 *
		 * // Now create an initialiase an audio buffer
		 * TAudioBuffer<double> audio;
		 * audio.initialise(format.m_numberOfSamples, format.m_numberOfChannels);
		 *
		 * // Now read in the audio data, in range  -1 to +1\n
		 * if (!waveFileFormat.readData(audio))
		 * {
		 *      printf("Failed to read data");
		 *      return;
		 * }
		 * @endcode
		 * \n
		 * \n
		 * WRITING\n
		 * \n
		 * @code
		 * // First we open the file
		 * TWaveFileFormat<double> waveFileFormat;
		 * if (!waveFileFormat.openFile(CFileStream::e_output, myFileName))
		 * {
		 *      printf("Failed to open file");
		 *      return;
		 * }
		 *
		 * // Now create the format to write to
		 * TWaveFileFormat<double>::SAudioFileFormat format;
		 * format.m_bitDepth         = 16;    // OR 8 OR 24
		 * format.m_sampleRate	     = 44100; // Anything you like
		 * format.m_numberOfChannels = 2;	  // Anything you like
		 * format.m_numberOfSamples	 = 1024;  // However many there are!
		 *
		 * // Now write the header
		 * if (!waveFileFormat.writeHeader(format))
		 * {
		 *      printf("Failed to write file header");
		 *      return;
		 * }
		 *
		 * // Now write the data
		 * if (!waveFileFormat.writeData(myAudioBuffer))
		 * {
		 *      printf("FAiled to write audio file");
		 *      return;
		 * }
		 * @endcode
		 *
		 * @date 13/04/2006
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: IAudioFileFormat.hpp,v 1.6 2007/02/08 21:08:09 paul Exp $
		 */
		template<typename TypeName> interface IAudioFileFormat
		{
		public:

//	===========================================================================

			/**
			 * @struct SAudioFileFormat IAudioFileFormat.hpp
			 * @brief Basic structure representing the format of the audio file
			 */
			struct SAudioFileFormat
			{
				unsigned long m_sampleRate;				/**< Sample rate of the audio */
				unsigned long m_numberOfChannels;		/**< Number of channels */
				unsigned long m_numberOfSamples;		/**< Number of samples */
				unsigned long m_bitDepth;				/**< Bit depth of the audio */
			};

//	===========================================================================

			/**
			 * @enum EBitDepth IAudioFileFormat.hpp
			 * @brief Represents currently supported bit depths
			 */
			enum EBitDepth
			{
				e_eightBit		= 8,			/**< 8 bit */
				e_sixteenBit	= 16,			/**< 16 bit */
				e_twentyFourBit = 24,			/**< 24 bit */
			};

//	===========================================================================

			/**
			 * Construction
			 */
			IAudioFileFormat() { }

			/**
			 * Destruction
			 */
			virtual ~IAudioFileFormat() { }

//	===========================================================================

			/**
			 * Open the file
			 * @param mode The mode of opening, read or write
			 * @param filename The name of the file to open
			 * @retval bool True if stream is opened
			 */
			virtual bool openFile(CFileStream::EStreamMode mode, const CSystemString &filename) = 0;

			/**
			 * Close the file
			 * @retval bool True if closed properly, false otherwise
			 */
			virtual bool closeFile() = 0;

//	===========================================================================

			/**
			 * Get the extension of the file format, each array entry sould contain one string
			 * @param array The array to fill in
			 */
			virtual void getFileExtension(TStringCountedPointerArray &array) const = 0;

			/**
			 * Check if a file is a valid format of this file format
			 * @param filename The name of the file to load
			 * @retval bool True if the file format matches (checks actual binary data, not just extension
			 */
			virtual bool isValidFormat(const CSystemString &filename) const = 0;

//	===========================================================================

			/**
			 * Read the header information and store a copy in the format
			 * @param format On return is filled with the values for this audio file
			 * @retval bool True if read the header correctly, false otherwise
			 */
			virtual bool readHeader(SAudioFileFormat &format) = 0;

			/**
			 * Read the entire audio file to the provided audio buffer
			 * @param data The data buffer to store te samples in
			 * @retval bool True if read correctly, false otherwise
			 */
			virtual bool readData(TAudioBuffer< TypeName > &data) = 0;

//	===========================================================================

			/**
			 * Write the header information and from the format
			 * @param format The format to write in
			 * @retval bool True if wrote the header correctly, false otherwise
			 */
			virtual bool writeHeader(const SAudioFileFormat &format) = 0;

			/**
			 * Write the entire audio file to the provided audio buffer
			 * @param data The data buffer to store
			 * @retval bool True if written correctly, false otherwise
			 */
			virtual bool writeData(const TAudioBuffer< TypeName > &data) = 0;

//	===========================================================================
		};
	}
}
#endif	// End of IAudioFileFormat.hpp